
export async function loadModificationData(siteId, defaultDrawingPrefix = 'FM_5_PF_') {
    // Early return for missing siteId
    if (!siteId) {
        return {
            lastModifiedDate: 'Select OLT first',
            lastModifiedBy: 'N/A',
            indexDrawing: `${defaultDrawingPrefix}OLT00_250000_0`
        };
    }

    try {
        const response = await fetch(`get_last_modification.php?siteid=${encodeURIComponent(siteId)}`);
        
        if (!response.ok) {
            throw new Error(`Server responded with status ${response.status}`);
        }
        
        const data = await response.json();
        
        if (!data?.success) {
            return getDefaultData(defaultDrawingPrefix);
        }

        // Process dates with 24-hour format
        const { displayDate, drawingDate } = processDates(data.updated_time);
        
        // Format site ID for drawing number
        const formattedSiteId = formatSiteId(siteId);
        
        return {
            lastModifiedBy: data.update_by || 'N/A',
            lastModifiedDate: displayDate,
            indexDrawing: generateDrawingNumber(defaultDrawingPrefix, formattedSiteId, drawingDate)
        };
        
    } catch (error) {
        console.error('Error loading modification data:', error);
        return getDefaultData(defaultDrawingPrefix);
    }
}

/**
 * Processes dates for display (24-hour format) and drawing number
 */
function processDates(dateString) {
    const fallbackDate = new Date();
    const dateObj = dateString && dateString !== 'N/A' 
        ? new Date(dateString) 
        : fallbackDate;

    // For display in 24-hour format (YYYY-MM-DD HH:MM)
    const displayDate = isNaN(dateObj.getTime()) 
        ? 'N/A'
        : dateObj.toLocaleString('en-US', {
            year: 'numeric',
            month: '2-digit',
            day: '2-digit',
            hour: '2-digit',
            minute: '2-digit',
            hour12: false
        }).replace(',', '');

    // For drawing number (YYMMDD)
    const year2Digit = dateObj.getFullYear().toString().slice(-2);
    const month = String(dateObj.getMonth() + 1).padStart(2, '0');
    const day = String(dateObj.getDate()).padStart(2, '0');
    const drawingDate = year2Digit + month + day;

    return { displayDate, drawingDate };
}

/**
 * Formats site ID for drawing number (first 3 + last 2 chars)
 */
function formatSiteId(siteId) {
    return siteId.length >= 5 
        ? siteId.substring(0, 3) + siteId.slice(-2)
        : siteId;
}

/**
 * Generates the drawing number string
 */
function generateDrawingNumber(prefix, siteId, date) {
    return `${prefix}${siteId}_${date}_0`;
}

/**
 * Returns default data when no valid data is available
 */
function getDefaultData(prefix) {
    return {
        lastModifiedDate: 'N/A',
        lastModifiedBy: 'N/A',
        indexDrawing: generateDrawingNumber(prefix, 'OLT00', '250000')
    };
}