/**
 * Module for managing OLT map display
 * 
 * @param {string} mapContainerId - ID of the map container element
 * @returns {Object} - Map module interface
 */
export function createMapModule(mapContainerId) {
    let map = null;
    let marker = null;
    let currentSite = '';

    // Initialize the map
    function initMap() {
        if (!document.getElementById(mapContainerId)) {
            console.error('Map container not found');
            return false;
        }

        map = L.map(mapContainerId, {
            zoomControl: false, 
            scrollWheelZoom: true
        }).setView([-2.5489, 118.0149], 5);
        
        L.tileLayer('https://{s}.basemaps.cartocdn.com/light_all/{z}/{x}/{y}.png').addTo(map);
        
        map.addControl(new L.Control.FullScreen({
            title: {
                'false': 'View Fullscreen',
                'true': 'Exit Fullscreen'
            },
            position: 'topright'
        }));

        return true;
    }

    // Show location on the map
    function showLocation(locationData, siteId) {
        if (!map && !initMap()) {
            return false;
        }

        currentSite = siteId;

        // Validate location data
        if (!locationData || !locationData.lat || !locationData.lng) {
            console.error('Invalid location data:', locationData);
            alert('OLT location not found in database');
            return false;
        }

        const lat = parseFloat(locationData.lat);
        const lng = parseFloat(locationData.lng);

        if (isNaN(lat) || isNaN(lng)) {
            console.error('Invalid coordinates:', locationData);
            return false;
        }

        const latLng = [lat, lng];

        // Remove old marker if exists
        if (marker) {
            map.removeLayer(marker);
        }

        // Create new marker
        marker = L.marker(latLng).addTo(map)
            .bindPopup(`<b>${currentSite}</b>`)
            .openPopup();

        // Set view to marker location
        map.setView(latLng, 15);
        return true;
    }

    return {
        showLocation
    };
}