<?php
require_once 'dbconfig.php';

header('Content-Type: application/json');

try {
    // Validate input
    if (!isset($_GET['siteid']) || empty(trim($_GET['siteid']))) {
        throw new Exception("Valid Site ID parameter is required");
    }

    $siteId = trim($_GET['siteid']);
    $db = getDBConnection();

    if (!$db || $db->connect_error) {
        throw new Exception("Database connection failed");
    }

    // Query to find direct connections
    $query = "SELECT DISTINCT siteid, farend, jenis2 
              FROM assetdata 
              WHERE (siteid = ? OR farend = ?)
              AND farend IS NOT NULL
              AND farend != ''";
              
    $stmt = $db->prepare($query);
    if (!$stmt) {
        throw new Exception("Database query preparation failed");
    }

    $stmt->bind_param('ss', $siteId, $siteId);
    if (!$stmt->execute()) {
        throw new Exception("Query execution failed");
    }

    $result = $stmt->get_result();
    $connections = $result->fetch_all(MYSQLI_ASSOC);

    $nodes = [];
    $edges = [];
    $nodeIds = [$siteId => true]; // Track unique nodes

    // Add main node with special styling
    $nodes[] = [
        'data' => [
            'id' => $siteId,
            'type' => 'current',
            'label' => $siteId,
            'device_type' => 'olt'
        ]
    ];

    foreach ($connections as $row) {
        $source = $row['siteid'];
        $target = $row['farend'];
        $deviceType = strtolower($row['jenis2'] ?? 'unknown');

        // Determine node type
        $nodeType = ($source === $siteId || $target === $siteId) 
            ? 'direct' 
            : 'indirect';

        // Process source node
        if (!isset($nodeIds[$source])) {
            $nodes[] = [
                'data' => [
                    'id' => $source,
                    'type' => $nodeType,
                    'label' => $source,
                    'device_type' => $deviceType
                ]
            ];
            $nodeIds[$source] = true;
        }

        // Process target node
        if (!isset($nodeIds[$target])) {
            $nodes[] = [
                'data' => [
                    'id' => $target,
                    'type' => $nodeType,
                    'label' => $target,
                    'device_type' => $deviceType
                ]
            ];
            $nodeIds[$target] = true;
        }

        // Add edge (direction: source -> target)
        if ($source && $target) {
            $edgeId = "{$source}-{$target}";
            $edges[$edgeId] = [
                'data' => [
                    'id' => $edgeId,
                    'source' => $source,
                    'target' => $target
                ]
            ];
        }
    }

    // Convert edges associative array to indexed array
    $uniqueEdges = array_values($edges);

    // Handle empty result case
    if (count($nodes) === 1 && empty($uniqueEdges)) {
        $nodes[] = [
            'data' => [
                'id' => 'no-connections',
                'type' => 'message',
                'label' => 'No network connections found',
                'device_type' => 'message'
            ]
        ];
    }

    echo json_encode([
        'success' => true,
        'data' => [
            'nodes' => $nodes,
            'edges' => $uniqueEdges
        ],
        'meta' => [
            'node_count' => count($nodes),
            'edge_count' => count($uniqueEdges)
        ]
    ]);

} catch (Exception $e) {
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'error' => $e->getMessage(),
        'trace' => $e->getTraceAsString()
    ]);
} finally {
    if (isset($db)) {
        $db->close();
    }
}