<?php
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

require_once 'dbcito.php';

// Parameter fixed source
$fixedSource = 'BYO-OLT-06';

try {
    // Query untuk mendapatkan semua target yang berelasi dengan AGS-OLT-01
    $query = "SELECT DISTINCT siteid as target 
              FROM assetdata 
              WHERE farend = ?";
    
    $stmt = $conn->prepare($query);
    $stmt->bind_param("s", $fixedSource);
    $stmt->execute();
    $result = $stmt->get_result();

    $nodes = [];
    $edges = [];
    $nodeIds = [];

    // Tambahkan fixed source node
    $nodes[] = [
        'data' => [
            'id' => $fixedSource,
            'label' => $fixedSource,
            'type' => 'olt',
            'weight' => 100
        ]
    ];
    $nodeIds[] = $fixedSource;

    if ($result->num_rows > 0) {
        while ($row = $result->fetch_assoc()) {
            $target = trim($row['target']);
            
            if (empty($target)) continue;
            
            // Tambahkan target node jika belum ada
            if (!in_array($target, $nodeIds)) {
                $nodes[] = [
                    'data' => [
                        'id' => $target,
                        'label' => $target,
                        'type' => 'site',
                        'weight' => 50
                    ]
                ];
                $nodeIds[] = $target;
            }
            
            // Tambahkan edge dari AGS-OLT-01 ke target
            $edges[] = [
                'data' => [
                    'id' => $fixedSource.'-'.$target,
                    'source' => $fixedSource,
                    'target' => $target,
                    'label' => 'Connected'
                ]
            ];
            
            // Query untuk mendapatkan hubungan sekunder dari target
            $secondaryQuery = "SELECT DISTINCT a.siteid as secondary_target 
                              FROM assetdata a
                              JOIN assetdata b ON a.farend = b.siteid
                              WHERE b.farend = ? AND a.siteid != ?";
            
            $secStmt = $conn->prepare($secondaryQuery);
            $secStmt->bind_param("ss", $fixedSource, $target);
            $secStmt->execute();
            $secResult = $secStmt->get_result();
            
            if ($secResult->num_rows > 0) {
                while ($secRow = $secResult->fetch_assoc()) {
                    $secTarget = trim($secRow['secondary_target']);
                    
                    if (!empty($secTarget) && !in_array($secTarget, $nodeIds)) {
                        $nodes[] = [
                            'data' => [
                                'id' => $secTarget,
                                'label' => $secTarget,
                                'type' => 'secondary',
                                'weight' => 30
                            ]
                        ];
                        $nodeIds[] = $secTarget;
                        
                        $edges[] = [
                            'data' => [
                                'id' => $target.'-'.$secTarget,
                                'source' => $target,
                                'target' => $secTarget,
                                'label' => 'Secondary'
                            ]
                        ];
                    }
                }
            }
            $secStmt->close();
        }
    } else {
        throw new Exception("Tidak ditemukan data untuk OLT: ".$fixedSource);
    }
    
    $elements = array_merge($nodes, $edges);
    $jsonElements = json_encode($elements, JSON_PRETTY_PRINT);
    
    if (json_last_error() !== JSON_ERROR_NONE) {
        throw new Exception("JSON encode error: ".json_last_error_msg());
    }

} catch (Exception $e) {
    die("<div style='color:red; padding:10px; border:1px solid #f00;'>Error: ".$e->getMessage()."</div>");
} finally {
    if (isset($stmt)) $stmt->close();
    if (isset($secStmt)) $secStmt->close();
    $conn->close();
}
?>

<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Visualisasi Jaringan AGS-OLT-01</title>
    <style>
        #cy {
            width: 100%;
            height: 700px;
            border: 1px solid #ddd;
            border-radius: 5px;
            background-color: #f9f9f9;
        }
        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            margin: 20px;
            line-height: 1.6;
            color: #333;
        }
        h1 {
            color: #2c3e50;
            border-bottom: 2px solid #3498db;
            padding-bottom: 10px;
        }
        .info-panel {
            background-color: #f8f9fa;
            padding: 15px;
            border-radius: 5px;
            margin-bottom: 20px;
            box-shadow: 0 2px 5px rgba(0,0,0,0.1);
        }
        .legend {
            display: flex;
            gap: 15px;
            margin-top: 10px;
        }
        .legend-item {
            display: flex;
            align-items: center;
            gap: 5px;
        }
        .legend-color {
            width: 15px;
            height: 15px;
            border-radius: 50%;
        }
        .stats {
            margin-top: 20px;
            padding: 10px;
            background-color: #e9ecef;
            border-radius: 5px;
        }
    </style>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/cytoscape/3.23.0/cytoscape.min.js"></script>
</head>
<body>
    <h1>Visualisasi Jaringan dari AGS-OLT-01</h1>
    
    <div class="info-panel">
        <h3>Informasi Jaringan</h3>
        <p>Visualisasi hubungan dari OLT <strong><?php echo htmlspecialchars($fixedSource); ?></strong> ke seluruh site yang terhubung.</p>
        
        <div class="legend">
            <div class="legend-item">
                <div class="legend-color" style="background-color:#e74c3c;"></div>
                <span>OLT</span>
            </div>
            <div class="legend-item">
                <div class="legend-color" style="background-color:#3498db;"></div>
                <span>Site Utama</span>
            </div>
            <div class="legend-item">
                <div class="legend-color" style="background-color:#2ecc71;"></div>
                <span>Site Sekunder</span>
            </div>
        </div>
        
        <div class="stats">
            <strong>Statistik:</strong>
            <ul>
                <li>Total Node: <?php echo count($nodes); ?></li>
                <li>Total Edge: <?php echo count($edges); ?></li>
                <li>Site Utama: <?php echo count(array_filter($nodes, function($n) { return $n['data']['type'] === 'site'; })); ?></li>
                <li>Site Sekunder: <?php echo count(array_filter($nodes, function($n) { return $n['data']['type'] === 'secondary'; })); ?></li>
            </ul>
        </div>
    </div>
    
    <div id="cy"></div>
    
    <script>
        document.addEventListener('DOMContentLoaded', function() {
            var elements = <?php echo $jsonElements; ?>;
            
            var cy = cytoscape({
                container: document.getElementById('cy'),
                elements: elements,
                
                style: [
                    // Style untuk OLT
                    {
                        selector: 'node[type = "olt"]',
                        style: {
                            'label': 'data(label)',
                            'background-color': '#e74c3c',
                            'width': 50,
                            'height': 50,
                            'color': '#fff',
                            'text-valign': 'center',
                            'text-halign': 'center',
                            'font-weight': 'bold',
                            'font-size': '14px',
                            'border-color': '#c0392b',
                            'border-width': 3,
                            'shape': 'star'
                        }
                    },
                    // Style untuk site utama
                    {
                        selector: 'node[type = "site"]',
                        style: {
                            'label': 'data(label)',
                            'background-color': '#3498db',
                            'width': 40,
                            'height': 40,
                            'color': '#fff',
                            'text-valign': 'center',
                            'text-halign': 'center',
                            'font-weight': 'bold',
                            'border-color': '#2980b9',
                            'border-width': 2
                        }
                    },
                    // Style untuk site sekunder
                    {
                        selector: 'node[type = "secondary"]',
                        style: {
                            'label': 'data(label)',
                            'background-color': '#2ecc71',
                            'width': 35,
                            'height': 35,
                            'color': '#fff',
                            'text-valign': 'center',
                            'text-halign': 'center',
                            'border-color': '#27ae60',
                            'border-width': 2
                        }
                    },
                    // Style untuk edge utama
                    {
                        selector: 'edge[source = "AGS-OLT-01"]',
                        style: {
                            'width': 4,
                            'line-color': '#e74c3c',
                            'curve-style': 'bezier',
                            'target-arrow-color': '#e74c3c',
                            'target-arrow-shape': 'triangle',
                            'arrow-scale': 1.5,
                            'label': 'data(label)',
                            'font-size': '12px',
                            'color': '#c0392b',
                            'text-rotation': 'autorotate',
                            'text-margin-y': -10,
                            'z-index': 999
                        }
                    },
                    // Style untuk edge sekunder
                    {
                        selector: 'edge',
                        style: {
                            'width': 2,
                            'line-color': '#95a5a6',
                            'curve-style': 'bezier',
                            'target-arrow-color': '#7f8c8d',
                            'target-arrow-shape': 'triangle',
                            'arrow-scale': 1.2,
                            'label': 'data(label)',
                            'font-size': '11px',
                            'color': '#34495e',
                            'text-rotation': 'autorotate'
                        }
                    },
                    // Style untuk hover
                    {
                        selector: 'node:selected',
                        style: {
                            'background-color': '#f39c12',
                            'border-color': '#d35400'
                        }
                    },
                    {
                        selector: 'edge:selected',
                        style: {
                            'line-color': '#f39c12',
                            'target-arrow-color': '#f39c12'
                        }
                    }
                ],
                
                layout: {
                    name: 'concentric',
                    concentric: function(node) {
                        if (node.data('type') === 'olt') {
                            return 10;
                        } else if (node.data('type') === 'site') {
                            return 5;
                        } else {
                            return 1;
                        }
                    },
                    levelWidth: function(nodes) {
                        return 2;
                    },
                    minNodeSpacing: 30,
                    padding: 50,
                    animate: true,
                    animationDuration: 1000,
                    fit: true
                }
            });
            
            // Fit jaringan ke layar
            cy.fit();
            
            // Event untuk node
            cy.on('tap', 'node', function(evt) {
                var node = evt.target;
                var degree = node.degree();
                console.log('Node: ' + node.id() + ', Degree: ' + degree);
                
                // Highlight node dan edge yang terhubung
                cy.elements().removeClass('highlight');
                node.addClass('highlight');
                node.connectedEdges().addClass('highlight');
                node.neighborhood().addClass('highlight');
            });
            
            // Event untuk edge
            cy.on('tap', 'edge', function(evt) {
                var edge = evt.target;
                console.log('Edge: ' + edge.id());
                
                // Highlight edge
                cy.elements().removeClass('highlight');
                edge.addClass('highlight');
                edge.source().addClass('highlight');
                edge.target().addClass('highlight');
            });
            
            // Tambahkan style untuk highlight
            cy.style()
                .selector('.highlight')
                .style({
                    'background-color': '#f39c12',
                    'line-color': '#f39c12',
                    'target-arrow-color': '#f39c12',
                    'border-color': '#d35400'
                })
                .update();
        });
    </script>
</body>
</html>